# Contributing to Softihub

Thank you for your interest in contributing to Softihub! This guide will help you add new sections to our platform.

## Table of Contents

- [Getting Started](#getting-started)
- [Section Guidelines](#section-guidelines)
- [Color System](#color-system)
- [Code Standards](#code-standards)
- [Styling Rules](#styling-rules)
- [Submission Process](#submission-process)

## Getting Started

This guide is for **Softihub team members** who have contributor access to add new sections.

### How to Contribute

1. **Login to Softihub** with your contributor account
2. Navigate to **"Add Section"** from your dashboard
3. Review existing sections for inspiration and consistency
4. Follow the guidelines below when creating your section
5. Submit your section for review
6. **Admin will review** and either approve or reject your submission
7. If rejected, you'll receive feedback on what needs to be improved

### Contributor Account

If you don't have a contributor account yet:

- Contact the admin team to request access
- You'll receive login credentials
- Once logged in, you can start adding sections immediately

## Section Guidelines

### Section Structure

Each section should include:

- **Name**: Clear, descriptive title (e.g., "Hero Section", "Pricing Cards")
- **Description**: Brief explanation of the section's purpose
- **Category**: Choose from existing categories or propose a new one
- **Tags**: Relevant keywords for searchability
- **Framework**: HTML, React, Vue, or Angular
- **Styling**: Tailwind CSS, CSS, or SCSS

### Code Quality

- ✅ Clean, readable code with proper indentation
- ✅ Semantic HTML5 elements
- ✅ Responsive design (mobile-first approach)
- ✅ Accessibility compliant (WCAG 2.1 AA)
- ✅ Cross-browser compatible
- ✅ No inline styles (use classes)
- ✅ Well-commented code for complex logic

## Color System

**IMPORTANT**: All sections MUST use our standardized color system. Never use hardcoded color values.

### Available Color Variables

#### Primary Colors

```css
--color-primary        /* Main brand color */
--color-secondary      /* Secondary brand color */
--color-accent         /* Accent/highlight color */
```

#### Semantic Colors

```css
--color-success        /* Success states (green) */
--color-warning        /* Warning states (yellow/orange) */
--color-danger         /* Error/danger states (red) */
--color-info           /* Informational states (blue) */
```

#### Neutral Scale

```css
--neutral-0            /* Pure white */
--neutral-50           /* Lightest gray */
--neutral-100
--neutral-200
--neutral-300
--neutral-400
--neutral-500          /* Mid gray */
--neutral-600
--neutral-700
--neutral-800
--neutral-900          /* Darkest gray */
--neutral-950          /* Almost black */
```

#### Images

**IMPORTANT**: All images in sections MUST use `placeholder.png` with proper width and height attributes.

**Why Width and Height Matter:**

- **Prevents Layout Shift**: Browser reserves space before image loads
- **Better Performance**: Improves Core Web Vitals (CLS score)
- **Faster Rendering**: Browser can calculate layout immediately
- **Better UX**: No content jumping when images load

**✅ DO THIS:**

```html
<!-- Always include width and height attributes -->
<img src="placeholder.png" alt="Descriptive alt text" width="800" height="600" class="w-full rounded-lg" />

<!-- For responsive images with aspect ratio -->
<img src="placeholder.png" alt="Hero banner image" width="1200" height="400" class="h-auto w-full rounded-lg" />

<!-- For icons and small images -->
<img src="placeholder.png" alt="Feature icon" width="64" height="64" class="rounded-md" />

<!-- For profile/avatar images -->
<img src="placeholder.png" alt="User profile picture" width="100" height="100" class="rounded-full" />
```

**❌ DON'T DO THIS:**

```html
<!-- Missing width/height - causes layout shift -->
<img src="placeholder.png" alt="Image" class="w-full" />

<!-- Using different image names -->
<img src="hero.jpg" width="800" height="600" alt="Hero" />
<img src="banner.png" width="1200" height="400" alt="Banner" />

<!-- Using inline styles for dimensions -->
<img src="placeholder.png" alt="Image" style="width: 800px; height: 600px;" />
```

**Common Image Dimensions:**

| Use Case      | Width | Height | Aspect Ratio |
| ------------- | ----- | ------ | ------------ |
| Hero Banner   | 1200  | 400    | 3:1          |
| Feature Image | 800   | 600    | 4:3          |
| Card Image    | 400   | 300    | 4:3          |
| Thumbnail     | 200   | 200    | 1:1          |
| Icon          | 64    | 64     | 1:1          |
| Avatar        | 100   | 100    | 1:1          |
| Blog Post     | 800   | 450    | 16:9         |
| Product Image | 600   | 600    | 1:1          |

**Best Practices:**

1. **Always use `placeholder.png`** - Never use actual image filenames
2. **Always include `width` and `height`** - Prevents layout shift
3. **Use descriptive `alt` text** - For accessibility
4. **Use Tailwind classes for styling** - `w-full`, `h-auto`, `rounded-lg`, etc.
5. **Maintain aspect ratio** - Use `h-auto` with `w-full` for responsive images
6. **Optimize for responsive** - Consider different sizes for mobile/desktop

**Responsive Image Example:**

```html
<!-- Desktop: 1200x400, Mobile: scales proportionally -->
<picture>
    <source media="(min-width: 768px)" srcset="placeholder.png" width="1200" height="400" />
    <img src="placeholder.png" alt="Responsive hero image" width="600" height="400" class="h-auto w-full rounded-lg" />
</picture>

<!-- Or use simple responsive approach -->
<img src="placeholder.png" alt="Responsive image" width="1200" height="600" class="h-auto w-full rounded-lg object-cover" />
```

### Tailwind CSS Color Usage

When using Tailwind CSS, use these custom color classes:

```html
<!-- Primary colors -->
<div class="border-primary bg-primary text-primary">
    <!-- Semantic colors -->
    <button class="bg-success text-success-foreground">Success</button>
    <div class="bg-warning text-warning-foreground">Warning</div>
    <div class="bg-danger text-danger-foreground">Error</div>

    <!-- Neutral colors -->
    <div class="bg-neutral-100 text-neutral-900">
        <div class="border-neutral-300">
            <!-- Background & text -->
            <div class="bg-background text-foreground">
                <p class="text-muted-foreground"></p>
            </div>
        </div>
    </div>
</div>
```

### ❌ DON'T DO THIS

```html
<!-- Never use hardcoded colors -->
<div class="bg-blue-500">
    ❌ Wrong
    <div class="text-red-600">
        ❌ Wrong
        <div style="color: #3b82f6">
            ❌ Wrong
            <div class="bg-[#ff0000]">❌ Wrong</div>
        </div>
    </div>
</div>
```

### ✅ DO THIS

```html
<!-- Always use color variables -->
<div class="bg-primary">
    ✅ Correct
    <div class="text-danger">
        ✅ Correct
        <div class="bg-neutral-100">✅ Correct</div>
    </div>
</div>
```

## Code Standards

### HTML Sections

```html
<!-- Good Example -->
<section class="bg-background py-16">
    <div class="container mx-auto px-4">
        <h2 class="mb-8 text-3xl font-bold text-foreground">Section Title</h2>
        <div class="grid grid-cols-1 gap-6 md:grid-cols-3">
            <div class="rounded-lg border border-neutral-200 bg-neutral-50 p-6">
                <h3 class="mb-4 text-xl font-semibold text-primary">Card Title</h3>
                <p class="text-muted-foreground">Card description text.</p>
                <button class="mt-4 rounded-md bg-primary px-6 py-2 text-white hover:bg-primary/90">Action</button>
            </div>
        </div>
    </div>
</section>
```

### HTML with Custom CSS and JavaScript

**IMPORTANT**: If your HTML section requires additional CSS or JavaScript, add them **BELOW** the HTML code in the following format:

#### Structure

```html
<!-- Your HTML Section Code -->
<section class="bg-background py-16">
    <div class="container mx-auto px-4">
        <h2 class="mb-8 text-3xl font-bold text-foreground">Interactive Section</h2>
        <div class="carousel" id="myCarousel">
            <!-- Carousel content -->
        </div>
    </div>
</section>

<!-- Custom CSS - Add below HTML -->
<style>
    /* Use CSS custom properties for colors */
    .carousel {
        background-color: var(--color-background);
        border: 1px solid var(--color-border);
    }

    .carousel-item {
        color: var(--color-foreground);
    }

    .carousel-item:hover {
        background-color: var(--color-primary);
        color: white;
    }

    /* Responsive styles */
    @media (max-width: 768px) {
        .carousel {
            padding: 1rem;
        }
    }
</style>

<!-- Custom JavaScript - Add below CSS -->
<script>
    // Your JavaScript code
    document.addEventListener('DOMContentLoaded', function () {
        const carousel = document.getElementById('myCarousel');

        // Your carousel logic here
        function initCarousel() {
            // Implementation
        }

        initCarousel();
    });
</script>
```

#### CSS Guidelines for Custom Styles

**✅ DO:**

```css
/* Use CSS custom properties */
.my-component {
    background-color: var(--color-primary);
    color: var(--color-foreground);
    border-color: var(--neutral-300);
}

/* Support dark mode */
.my-component:hover {
    background-color: var(--color-primary);
    opacity: 0.9;
}

/* Use relative units */
.my-component {
    padding: 1rem;
    margin: 0.5rem;
    font-size: 1rem;
}

/* Add smooth transitions */
.my-component {
    transition: all 0.3s ease;
}
```

**❌ DON'T:**

```css
/* Never use hardcoded colors */
.my-component {
    background-color: #3b82f6; /* ❌ Wrong */
    color: #ffffff; /* ❌ Wrong */
    border: 1px solid #e5e7eb; /* ❌ Wrong */
}

/* Don't use fixed pixel values for everything */
.my-component {
    width: 500px; /* ❌ Use responsive units */
    height: 300px; /* ❌ Use responsive units */
}

/* Don't use !important unless absolutely necessary */
.my-component {
    color: red !important; /* ❌ Avoid */
}
```

#### JavaScript Guidelines

**✅ DO:**

```javascript
// Use modern JavaScript (ES6+)
document.addEventListener('DOMContentLoaded', function () {
    const elements = document.querySelectorAll('.my-element');

    elements.forEach((element) => {
        element.addEventListener('click', handleClick);
    });
});

// Use descriptive function names
function handleClick(event) {
    event.preventDefault();
    // Your logic here
}

// Clean up event listeners when needed
function cleanup() {
    elements.forEach((element) => {
        element.removeEventListener('click', handleClick);
    });
}
```

**❌ DON'T:**

```javascript
// Don't use inline event handlers
// <button onclick="doSomething()">  ❌ Wrong

// Don't pollute global namespace
var myVar = 'value'; // ❌ Use const/let in functions

// Don't use jQuery unless absolutely necessary
$('.my-element').click(function () {
    // ❌ Use vanilla JS
    // ...
});
```

#### Complete Example with CSS and JS

```html
<!-- HTML Section -->
<section class="bg-background py-16">
    <div class="container mx-auto px-4">
        <h2 class="mb-8 text-3xl font-bold text-foreground">Accordion Section</h2>
        <div class="accordion">
            <div class="accordion-item mb-4 rounded-lg border border-neutral-200">
                <button class="accordion-header w-full p-4 text-left font-semibold text-foreground transition-colors hover:bg-neutral-50">What is Softihub?</button>
                <div class="accordion-content hidden p-4 text-muted-foreground">Softihub is a platform for storing and managing code snippets.</div>
            </div>
            <div class="accordion-item mb-4 rounded-lg border border-neutral-200">
                <button class="accordion-header w-full p-4 text-left font-semibold text-foreground transition-colors hover:bg-neutral-50">How do I contribute?</button>
                <div class="accordion-content hidden p-4 text-muted-foreground">Check our CONTRIBUTING.md file for detailed guidelines.</div>
            </div>
        </div>
    </div>
</section>

<!-- Custom CSS -->
<style>
    .accordion-item {
        background-color: var(--color-background);
        transition: all 0.3s ease;
    }

    .accordion-item:hover {
        border-color: var(--color-primary);
    }

    .accordion-header {
        cursor: pointer;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .accordion-header::after {
        content: '+';
        font-size: 1.5rem;
        color: var(--color-primary);
        transition: transform 0.3s ease;
    }

    .accordion-header.active::after {
        transform: rotate(45deg);
    }

    .accordion-content {
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.3s ease;
    }

    .accordion-content.active {
        display: block;
        max-height: 500px;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
        .accordion-header {
            font-size: 0.875rem;
            padding: 0.75rem;
        }
    }
</style>

<!-- Custom JavaScript -->
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const accordionHeaders = document.querySelectorAll('.accordion-header');

        accordionHeaders.forEach((header) => {
            header.addEventListener('click', function () {
                const content = this.nextElementSibling;
                const isActive = content.classList.contains('active');

                // Close all accordion items
                document.querySelectorAll('.accordion-content').forEach((item) => {
                    item.classList.remove('active');
                });

                document.querySelectorAll('.accordion-header').forEach((item) => {
                    item.classList.remove('active');
                });

                // Toggle current item
                if (!isActive) {
                    content.classList.add('active');
                    this.classList.add('active');
                }
            });
        });
    });
</script>
```

#### Key Points for Custom CSS/JS

1. **Always use CSS custom properties** for colors (never hardcoded values)
2. **Place CSS after HTML** in a `<style>` tag
3. **Place JavaScript after CSS** in a `<script>` tag
4. **Use vanilla JavaScript** (no jQuery unless absolutely necessary)
5. **Wrap JS in DOMContentLoaded** to ensure DOM is ready
6. **Add comments** to explain complex logic
7. **Keep it minimal** - only add what's necessary
8. **Test thoroughly** across different browsers
9. **Ensure accessibility** - keyboard navigation, ARIA labels
10. **Make it responsive** - test on mobile, tablet, desktop

### React/TSX Sections

```tsx
// Good Example
export function HeroSection() {
    return (
        <section className="bg-background py-16">
            <div className="container mx-auto px-4">
                <h1 className="text-foreground mb-4 text-4xl font-bold">Hero Title</h1>
                <p className="text-muted-foreground mb-8 text-lg">Hero description</p>
                <button className="bg-primary hover:bg-primary/90 rounded-lg px-8 py-3 text-white transition-colors">Get Started</button>
            </div>
        </section>
    );
}
```

## Styling Rules

### 1. Spacing

Use consistent spacing scale:

- `p-2, p-4, p-6, p-8, p-12, p-16` for padding
- `m-2, m-4, m-6, m-8, m-12, m-16` for margin
- `gap-2, gap-4, gap-6, gap-8` for grid/flex gaps

### 2. Typography

```html
<!-- Headings -->
<h1 class="text-4xl font-bold text-foreground">
    <h2 class="text-3xl font-bold text-foreground">
        <h3 class="text-2xl font-semibold text-foreground">
            <!-- Body text -->
            <p class="text-base text-foreground"></p>
            <p class="text-sm text-muted-foreground"></p>
        </h3>
    </h2>
</h1>
```

### 3. Borders & Shadows

```html
<!-- Borders -->
<div class="rounded-lg border border-neutral-200">
    <div class="rounded-md border-2 border-primary">
        <!-- Shadows -->
        <div class="shadow-sm">
            <!-- Subtle -->
            <div class="shadow-md">
                <!-- Medium -->
                <div class="shadow-lg"><!-- Large --></div>
            </div>
        </div>
    </div>
</div>
```

### 4. Hover States

Always include hover states for interactive elements:

```html
<button class="bg-primary transition-colors hover:bg-primary/90">
    <a class="text-primary transition-colors hover:text-primary/80"> <div class="border-neutral-200 transition-colors hover:border-primary"></div></a>
</button>
```

### 5. Responsive Design

Use mobile-first approach:

```html
<div class="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-3">
    <div class="text-sm md:text-base lg:text-lg">
        <div class="p-4 md:p-6 lg:p-8"></div>
    </div>
</div>
```

### 6. Dark Mode Support

Ensure your section works in both light and dark modes:

```html
<!-- The color system handles this automatically -->
<div class="bg-background text-foreground">
    <div class="bg-neutral-100 dark:bg-neutral-800"></div>
</div>
```

## Accessibility Requirements

### Required Attributes

- ✅ Use semantic HTML (`<header>`, `<nav>`, `<main>`, `<section>`, `<article>`)
- ✅ Add `alt` text to all images
- ✅ Include `aria-label` for icon-only buttons
- ✅ Ensure proper heading hierarchy (h1 → h2 → h3)
- ✅ Maintain color contrast ratio of at least 4.5:1
- ✅ Make all interactive elements keyboard accessible

### Example

```html
<button aria-label="Close menu" class="rounded-md bg-primary p-2 text-white">
    <svg aria-hidden="true">...</svg>
</button>

<img src="placeholder.png" alt="Team collaboration in modern office" class="w-full rounded-lg" />
```

## Performance Best Practices

- ✅ Optimize images (use WebP format when possible)
- ✅ Lazy load images below the fold
- ✅ Minimize use of custom fonts
- ✅ Avoid excessive animations
- ✅ Keep HTML structure simple and clean

## Submission Process

### 1. Test Your Section

Before submitting, ensure you've tested your section thoroughly:

- [ ] Test on multiple screen sizes (mobile, tablet, desktop)
- [ ] Test in different browsers (Chrome, Firefox, Safari)
- [ ] Test in both light and dark modes
- [ ] Verify all colors use the color system
- [ ] Check accessibility with screen reader
- [ ] Validate HTML (use W3C Validator)
- [ ] Test all interactive elements (buttons, forms, etc.)
- [ ] Verify custom CSS and JavaScript work correctly

### 2. Submit Through Platform

1. **Login** to your contributor account
2. Navigate to **"Add Section"** or **"My Sections"**
3. Click **"Create New Section"**
4. Fill in the section details:
    - **Name**: Clear, descriptive title
    - **Description**: Brief explanation of the section's purpose
    - **Category**: Select appropriate category
    - **Tags**: Add relevant keywords
    - **Framework**: Choose HTML, React, Vue, or Angular
    - **Styling**: Select Tailwind CSS, CSS, or SCSS

5. **Add your code**:
    - Paste your HTML code in the code editor
    - If using custom CSS/JS, add them below the HTML
    - Use the preview feature to verify appearance

6. **Upload screenshots** (if required):
    - Light mode screenshot
    - Dark mode screenshot
    - Mobile view (optional but recommended)

7. Click **"Submit for Review"**

### 3. Review Process

Once submitted:

1. **Admin receives notification** of your submission
2. **Admin reviews** your section against guidelines:
    - Code quality and standards
    - Color system compliance
    - Accessibility requirements
    - Responsive design
    - Overall quality and usefulness

3. **Admin decision**:
    - ✅ **Approved**: Section goes live immediately
    - ❌ **Rejected**: You receive detailed feedback

4. **If rejected**:
    - Review the admin's feedback carefully
    - Make necessary improvements
    - Resubmit for review

### 4. Submission Checklist

Before clicking "Submit for Review", confirm:

- [ ] Uses color system variables (no hardcoded colors)
- [ ] Responsive design (mobile, tablet, desktop)
- [ ] Accessibility compliant (WCAG 2.1 AA)
- [ ] Dark mode compatible
- [ ] Clean, well-commented code
- [ ] Tested in multiple browsers
- [ ] Follows spacing and typography guidelines
- [ ] Custom CSS uses CSS custom properties
- [ ] JavaScript is vanilla (no jQuery unless necessary)
- [ ] All images are optimized
- [ ] Section name and description are clear
- [ ] Tested in multiple browsers
- [ ] Follows spacing and typography guidelines

````

## Common Mistakes to Avoid

### ❌ Don't

```html
<!-- Hardcoded colors -->
<div class="bg-blue-500 text-white">
    <!-- Inline styles -->
    <div style="padding: 20px; color: red;">
        <!-- Poor contrast -->
        <p class="text-neutral-300">Text on white background</p>

        <!-- Missing alt text -->
        <img src="placeholder.png" />

        <!-- Non-semantic HTML -->
        <div class="header">
            <div class="nav"></div>
        </div>
    </div>
</div>
````

### ✅ Do

```html
<!-- Use color system -->
<div class="bg-primary text-white">
    <!-- Use classes -->
    <div class="text-danger p-5">
        <!-- Good contrast -->
        <p class="text-neutral-700 dark:text-neutral-300">Text on background</p>

        <!-- Include alt text -->
        <img src="placeholder.png" alt="Descriptive text" />

        <!-- Semantic HTML -->
        <header>
            <nav></nav>
        </header>
    </div>
</div>
```

## Need Help?

- 📖 Check existing sections for examples
- 💬 Open a discussion in GitHub Discussions
- 🐛 Report issues in GitHub Issues
- 📧 Contact maintainers

## Code of Conduct

- Be respectful and constructive
- Follow the guidelines
- Help others learn
- Give credit where due

---

Thank you for contributing to Softihub! Your sections help developers build better websites faster. 🚀
