<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Section;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SectionController extends Controller
{
    /**
     * Display all sections (pending, approved, rejected)
     */
    public function index(Request $request)
    {
        $query = Section::with(['user', 'approver', 'versions', 'category']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        $sections = $query->latest()->paginate($request->get('per_page', 20));

        // Get statistics
        $stats = [
            'total' => Section::count(),
            'pending' => Section::where('status', 'pending')->count(),
            'approved' => Section::where('status', 'approved')->count(),
            'rejected' => Section::where('status', 'rejected')->count(),
        ];

        return Inertia::render('admin/sections/index', [
            'sections' => $sections,
            'stats' => $stats,
            'filters' => [
                'status' => $request->status,
                'category' => $request->category,
                'search' => $request->search,
            ],
        ]);
    }

    /**
     * Show section details for review
     */
    public function show(Section $section)
    {
        $section->load(['user', 'approver', 'versions', 'category']);

        return Inertia::render('admin/sections/show', [
            'section' => $section,
        ]);
    }

    /**
     * Approve a section
     */
    public function approve(Request $request, Section $section)
    {
        if (! $request->user()->canApprove()) {
            abort(403, 'Unauthorized');
        }

        $section->approve($request->user()->id);

        return back()->with('success', 'Section approved successfully!');
    }

    /**
     * Reject a section
     */
    public function reject(Request $request, Section $section)
    {
        if (! $request->user()->canApprove()) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $section->reject($request->user()->id, $validated['reason']);

        return back()->with('success', 'Section rejected!');
    }

    /**
     * Delete a section
     */
    public function destroy(Section $section)
    {
        $section->delete();

        return back()->with('success', 'Section deleted successfully!');
    }

    /**
     * Bulk approve sections
     */
    public function bulkApprove(Request $request)
    {
        if (! $request->user()->canApprove()) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'section_ids' => 'required|array',
            'section_ids.*' => 'exists:sections,id',
        ]);

        $sections = Section::whereIn('id', $validated['section_ids'])->get();

        foreach ($sections as $section) {
            $section->approve($request->user()->id);
        }

        return back()->with('success', count($sections) . ' sections approved successfully!');
    }

    /**
     * Bulk reject sections
     */
    public function bulkReject(Request $request)
    {
        if (! $request->user()->canApprove()) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'section_ids' => 'required|array',
            'section_ids.*' => 'exists:sections,id',
            'reason' => 'required|string|max:1000',
        ]);

        $sections = Section::whereIn('id', $validated['section_ids'])->get();

        foreach ($sections as $section) {
            $section->reject($request->user()->id, $validated['reason']);
        }

        return back()->with('success', count($sections) . ' sections rejected!');
    }
}
