<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Section;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();

        $stats = [];

        if ($user->isAdmin()) {
            // Admin sees global statistics
            $stats = [
                'totalSections' => Section::count(),
                'approvedSections' => Section::where('status', 'approved')->count(),
                'pendingSections' => Section::where('status', 'pending')->count(),
                'rejectedSections' => Section::where('status', 'rejected')->count(),
                'totalCategories' => Category::count(),
                'totalViews' => Section::sum('views_count'),
                'totalCopies' => Section::sum('copies_count'),
                'totalUsers' => User::count(),
                'totalContributors' => User::where('role', 'contributor')->count(),
            ];
        } elseif ($user->isContributor()) {
            // Contributor sees their own statistics
            $accessibleTypes = $user->getAccessibleCategoryTypes();

            $stats = [
                'mySections' => Section::where('user_id', $user->id)->count(),
                'myApprovedSections' => Section::where('user_id', $user->id)
                    ->where('status', 'approved')
                    ->count(),
                'myPendingSections' => Section::where('user_id', $user->id)
                    ->where('status', 'pending')
                    ->count(),
                'myRejectedSections' => Section::where('user_id', $user->id)
                    ->where('status', 'rejected')
                    ->count(),
                'myCategories' => Category::where('user_id', $user->id)->count(),
                'myTotalViews' => (int) Section::where('user_id', $user->id)->sum('views_count'),
                'myTotalCopies' => (int) Section::where('user_id', $user->id)->sum('copies_count'),
                // Approved sections filtered by access type
                'approvedSections' => Section::where('status', 'approved')
                    ->whereHas('category', function ($query) use ($accessibleTypes) {
                        $query->whereIn('type', $accessibleTypes);
                    })
                    ->count(),
            ];
        } else {
            // Regular users see public statistics filtered by access type
            $accessibleTypes = $user->getAccessibleCategoryTypes();

            $stats = [
                'approvedSections' => Section::where('status', 'approved')
                    ->whereHas('category', function ($query) use ($accessibleTypes) {
                        $query->whereIn('type', $accessibleTypes);
                    })
                    ->count(),
                'totalViews' => Section::sum('views_count'),
                'totalCopies' => Section::sum('copies_count'),
            ];
        }

        return Inertia::render('dashboard', [
            'stats' => $stats,
        ]);
    }
}
