import { ConfirmationDialog } from '@/components/confirmation-dialog';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from '@/components/ui/table';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Edit, Plus, Trash2 } from 'lucide-react';
import { useState } from 'react';
import { toast } from 'sonner';

interface Category {
    id: number;
    name: string;
    slug: string;
    type: string;
    description: string | null;
    icon: string | null;
    color: string | null;
    sections_count: number;
    created_at: string;
    updated_at: string;
}

interface PaginatedCategories {
    data: Category[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
}

interface Props {
    categories: PaginatedCategories;
}

export default function CategoriesIndex({ categories }: Props) {
    const [deleteDialog, setDeleteDialog] = useState<{
        open: boolean;
        categoryId: number | null;
        categoryName: string;
    }>({
        open: false,
        categoryId: null,
        categoryName: '',
    });

    const handleDeleteClick = (categoryId: number, categoryName: string, sectionsCount: number) => {
        if (sectionsCount > 0) {
            toast.error(`Cannot delete "${categoryName}" because it has ${sectionsCount} sections.`);
            return;
        }

        setDeleteDialog({
            open: true,
            categoryId,
            categoryName,
        });
    };

    const handleDeleteConfirm = () => {
        if (!deleteDialog.categoryId) return;

        router.delete(`/admin/categories/${deleteDialog.categoryId}`, {
            onSuccess: () => {
                toast.success('Category deleted successfully!');
                setDeleteDialog({ open: false, categoryId: null, categoryName: '' });
            },
            onError: () => {
                toast.error('Failed to delete category');
            },
        });
    };

    return (
        <AppLayout>
            <Head title="Manage Categories" />

            <div className="container mx-auto px-4 py-8">
                {/* Header */}
                <div className="mb-8 flex items-center justify-between">
                    <div>
                        <h1 className="text-4xl font-bold mb-2">Manage Categories</h1>
                        <p className="text-muted-foreground">
                            Organize your component library with categories
                        </p>
                    </div>
                    <Link href="/admin/categories/create">
                        <Button>
                            <Plus className="mr-2 h-4 w-4" />
                            Add Category
                        </Button>
                    </Link>
                </div>

                {/* Stats */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Total Categories</CardDescription>
                            <CardTitle className="text-3xl">{categories.total}</CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Total Sections</CardDescription>
                            <CardTitle className="text-3xl">
                                {categories.data.reduce((sum, cat) => sum + cat.sections_count, 0)}
                            </CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Avg Sections/Category</CardDescription>
                            <CardTitle className="text-3xl">
                                {categories.total > 0
                                    ? Math.round(
                                        categories.data.reduce((sum, cat) => sum + cat.sections_count, 0) /
                                        categories.total
                                    )
                                    : 0}
                            </CardTitle>
                        </CardHeader>
                    </Card>
                </div>

                {/* Categories Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Categories</CardTitle>
                        <CardDescription>
                            Showing {categories.data.length} of {categories.total} categories
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Name</TableHead>
                                    <TableHead>Slug</TableHead>
                                    <TableHead>Type</TableHead>
                                    <TableHead>Description</TableHead>
                                    <TableHead>Icon</TableHead>
                                    <TableHead>Color</TableHead>
                                    <TableHead className="text-right">Sections</TableHead>
                                    <TableHead className="text-right">Actions</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {categories.data.map((category) => (
                                    <TableRow key={category.id}>
                                        <TableCell className="font-medium">{category.name}</TableCell>
                                        <TableCell>
                                            <Badge variant="outline">{category.slug}</Badge>
                                        </TableCell>
                                        <TableCell>
                                            <Badge variant={category.type === 'web' ? 'default' : 'secondary'}>
                                                {category.type === 'web' ? 'Web' : 'Dashboard'}
                                            </Badge>
                                        </TableCell>
                                        <TableCell className="max-w-md truncate">
                                            {category.description || '-'}
                                        </TableCell>
                                        <TableCell>{category.icon || '-'}</TableCell>
                                        <TableCell>
                                            {category.color ? (
                                                <div className="flex items-center gap-2">
                                                    <div
                                                        className="w-6 h-6 rounded border"
                                                        style={{ backgroundColor: category.color }}
                                                    />
                                                    <span className="text-xs text-muted-foreground">
                                                        {category.color}
                                                    </span>
                                                </div>
                                            ) : (
                                                '-'
                                            )}
                                        </TableCell>
                                        <TableCell className="text-right">
                                            <Badge>{category.sections_count}</Badge>
                                        </TableCell>
                                        <TableCell className="text-right">
                                            <div className="flex justify-end gap-2">
                                                <Link href={`/admin/categories/${category.id}/edit`}>
                                                    <Button size="sm" variant="outline">
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                </Link>
                                                <Button
                                                    size="sm"
                                                    variant="destructive"
                                                    onClick={() =>
                                                        handleDeleteClick(
                                                            category.id,
                                                            category.name,
                                                            category.sections_count
                                                        )
                                                    }
                                                    disabled={category.sections_count > 0}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </CardContent>
                </Card>
            </div>

            <ConfirmationDialog
                open={deleteDialog.open}
                onOpenChange={(open) => setDeleteDialog({ ...deleteDialog, open })}
                onConfirm={handleDeleteConfirm}
                title="Delete Category"
                description={`Are you sure you want to delete "${deleteDialog.categoryName}"? This action cannot be undone.`}
                confirmText="Delete"
                cancelText="Cancel"
                variant="destructive"
            />
        </AppLayout>
    );
}
