import { ConfirmationDialog } from '@/components/confirmation-dialog';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle
} from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from '@/components/ui/table';
import { Textarea } from '@/components/ui/textarea';
import AppLayout from '@/layouts/app-layout';
import type { AdminSectionStats, PaginatedSections, Section, SectionFilters } from '@/types/section';
import { Head, router, usePage } from '@inertiajs/react';
import { Check, Eye, Search, Trash2, X } from 'lucide-react';
import { useEffect, useState } from 'react';
import { toast } from 'sonner';

interface Props {
    sections: PaginatedSections;
    stats: AdminSectionStats;
    filters: SectionFilters;
}

export default function AdminSectionsIndex({ sections, stats, filters }: Props) {
    const { props } = usePage<{ success?: string }>();
    const [search, setSearch] = useState(filters.search || '');
    const [selectedStatus, setSelectedStatus] = useState(filters.category || 'all');
    const [rejectDialogOpen, setRejectDialogOpen] = useState(false);
    const [selectedSection, setSelectedSection] = useState<Section | null>(null);
    const [rejectionReason, setRejectionReason] = useState('');
    const [deleteDialog, setDeleteDialog] = useState<{
        open: boolean;
        section: Section | null;
    }>({
        open: false,
        section: null,
    });
    const [approveDialog, setApproveDialog] = useState<{
        open: boolean;
        section: Section | null;
    }>({
        open: false,
        section: null,
    });

    // Show flash messages
    useEffect(() => {
        if (props.success) {
            toast.success(props.success);
        }
    }, [props.success]);

    const handleSearch = () => {
        router.get(
            '/admin/sections',
            {
                search: search || undefined,
                status: selectedStatus !== 'all' ? selectedStatus : undefined,
            },
            { preserveState: true }
        );
    };

    const handleStatusChange = (value: string) => {
        setSelectedStatus(value);
        router.get(
            '/admin/sections',
            {
                search: search || undefined,
                status: value !== 'all' ? value : undefined,
            },
            { preserveState: true }
        );
    };

    const handleApproveClick = (section: Section) => {
        setApproveDialog({
            open: true,
            section,
        });
    };

    const handleApproveConfirm = () => {
        if (!approveDialog.section) return;

        router.post(`/admin/sections/${approveDialog.section.id}/approve`, {}, {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Section approved successfully!');
            },
            onError: () => {
                toast.error('Failed to approve section');
            },
        });
    };

    const handleReject = () => {
        if (!selectedSection || !rejectionReason.trim()) {
            toast.error('Please provide a rejection reason');
            return;
        }

        router.post(`/admin/sections/${selectedSection.id}/reject`,
            { reason: rejectionReason },
            {
                preserveScroll: true,
                onSuccess: () => {
                    toast.success('Section rejected!');
                    setRejectDialogOpen(false);
                    setRejectionReason('');
                    setSelectedSection(null);
                },
                onError: () => {
                    toast.error('Failed to reject section');
                },
            }
        );
    };

    const handleDeleteClick = (section: Section) => {
        setDeleteDialog({
            open: true,
            section,
        });
    };

    const handleDeleteConfirm = () => {
        if (!deleteDialog.section) return;

        router.delete(`/admin/sections/${deleteDialog.section.id}`, {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Section deleted successfully!');
            },
            onError: () => {
                toast.error('Failed to delete section');
            },
        });
    };

    const getStatusBadge = (status: string) => {
        switch (status) {
            case 'approved':
                return <Badge className="bg-green-500">Approved</Badge>;
            case 'pending':
                return <Badge className="bg-yellow-500">Pending</Badge>;
            case 'rejected':
                return <Badge className="bg-red-500">Rejected</Badge>;
            default:
                return <Badge>{status}</Badge>;
        }
    };

    return (
        <AppLayout>
            <Head title="Manage Sections" />

            <div className="container mx-auto px-4 py-8">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-4xl font-bold mb-2">Manage Sections</h1>
                    <p className="text-muted-foreground">Review and manage submitted components</p>
                </div>

                {/* Stats */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Total</CardDescription>
                            <CardTitle className="text-3xl">{stats.total}</CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Pending</CardDescription>
                            <CardTitle className="text-3xl text-yellow-500">{stats.pending}</CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Approved</CardDescription>
                            <CardTitle className="text-3xl text-green-500">{stats.approved}</CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Rejected</CardDescription>
                            <CardTitle className="text-3xl text-red-500">{stats.rejected}</CardTitle>
                        </CardHeader>
                    </Card>
                </div>

                {/* Filters */}
                <div className="mb-6 flex flex-col gap-4 md:flex-row md:items-center">
                    <div className="flex-1 flex gap-2">
                        <Input
                            placeholder="Search sections..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            className="max-w-md"
                        />
                        <Button onClick={handleSearch} size="icon">
                            <Search className="h-4 w-4" />
                        </Button>
                    </div>

                    <Select value={selectedStatus} onValueChange={handleStatusChange}>
                        <SelectTrigger className="w-[180px]">
                            <SelectValue placeholder="Status" />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="all">All Status</SelectItem>
                            <SelectItem value="pending">Pending</SelectItem>
                            <SelectItem value="approved">Approved</SelectItem>
                            <SelectItem value="rejected">Rejected</SelectItem>
                        </SelectContent>
                    </Select>
                </div>

                {/* Sections Table */}
                <Card>
                    <CardContent className="p-0">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Name</TableHead>
                                    <TableHead>Category</TableHead>
                                    <TableHead>Author</TableHead>
                                    <TableHead>Status</TableHead>
                                    <TableHead>Versions</TableHead>
                                    <TableHead>Created</TableHead>
                                    <TableHead className="text-right">Actions</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {sections.data.map((section) => (
                                    <TableRow key={section.id}>
                                        <TableCell className="font-medium">{section.name}</TableCell>
                                        <TableCell>
                                            <Badge variant="secondary">{section.category?.name}</Badge>
                                        </TableCell>
                                        <TableCell>{section.user?.name}</TableCell>
                                        <TableCell>{getStatusBadge(section.status)}</TableCell>
                                        <TableCell>
                                            <div className="flex gap-1">
                                                {Array.from(
                                                    new Set(section.versions?.map((v) => v.framework) || [])
                                                ).map((framework) => (
                                                    <Badge key={framework} variant="outline" className="text-xs">
                                                        {framework}
                                                    </Badge>
                                                ))}
                                            </div>
                                        </TableCell>
                                        <TableCell>
                                            {new Date(section.created_at).toLocaleDateString()}
                                        </TableCell>
                                        <TableCell className="text-right">
                                            <div className="flex justify-end gap-2">
                                                <Button
                                                    size="sm"
                                                    variant="outline"
                                                    onClick={() => router.visit(`/sections/${section.slug}`)}
                                                >
                                                    <Eye className="h-4 w-4" />
                                                </Button>

                                                {section.status === 'pending' && (
                                                    <>
                                                        <Button
                                                            size="sm"
                                                            variant="default"
                                                            onClick={() => handleApproveClick(section)}
                                                        >
                                                            <Check className="h-4 w-4" />
                                                        </Button>
                                                        <Button
                                                            size="sm"
                                                            variant="destructive"
                                                            onClick={() => {
                                                                setSelectedSection(section);
                                                                setRejectDialogOpen(true);
                                                            }}
                                                        >
                                                            <X className="h-4 w-4" />
                                                        </Button>
                                                    </>
                                                )}

                                                <Button
                                                    size="sm"
                                                    variant="destructive"
                                                    onClick={() => handleDeleteClick(section)}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </CardContent>
                </Card>

                {/* Pagination */}
                {sections.last_page > 1 && (
                    <div className="flex justify-center gap-2 mt-6">
                        <Button
                            variant="outline"
                            onClick={() =>
                                router.get('/admin/sections', {
                                    page: sections.current_page - 1,
                                    search: search || undefined,
                                    status: selectedStatus !== 'all' ? selectedStatus : undefined,
                                })
                            }
                            disabled={sections.current_page === 1}
                        >
                            Previous
                        </Button>

                        <div className="flex items-center gap-2">
                            Page {sections.current_page} of {sections.last_page}
                        </div>

                        <Button
                            variant="outline"
                            onClick={() =>
                                router.get('/admin/sections', {
                                    page: sections.current_page + 1,
                                    search: search || undefined,
                                    status: selectedStatus !== 'all' ? selectedStatus : undefined,
                                })
                            }
                            disabled={sections.current_page === sections.last_page}
                        >
                            Next
                        </Button>
                    </div>
                )}
            </div>

            {/* Reject Dialog */}
            <Dialog open={rejectDialogOpen} onOpenChange={setRejectDialogOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Reject Section</DialogTitle>
                        <DialogDescription>
                            Please provide a reason for rejecting "{selectedSection?.name}"
                        </DialogDescription>
                    </DialogHeader>

                    <Textarea
                        placeholder="Enter rejection reason..."
                        value={rejectionReason}
                        onChange={(e) => setRejectionReason(e.target.value)}
                        rows={4}
                    />

                    <DialogFooter>
                        <Button variant="outline" onClick={() => setRejectDialogOpen(false)}>
                            Cancel
                        </Button>
                        <Button variant="destructive" onClick={handleReject}>
                            Reject
                        </Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            <ConfirmationDialog
                open={approveDialog.open}
                onOpenChange={(open) => setApproveDialog({ ...approveDialog, open })}
                onConfirm={handleApproveConfirm}
                title="Approve Section"
                description={`Are you sure you want to approve "${approveDialog.section?.name}"? This will make it publicly visible to all users.`}
                confirmText="Approve"
                cancelText="Cancel"
                variant="default"
            />

            <ConfirmationDialog
                open={deleteDialog.open}
                onOpenChange={(open) => setDeleteDialog({ ...deleteDialog, open })}
                onConfirm={handleDeleteConfirm}
                title="Delete Section"
                description={`Are you sure you want to delete "${deleteDialog.section?.name}"? This action cannot be undone.`}
                confirmText="Delete"
                cancelText="Cancel"
                variant="destructive"
            />
        </AppLayout>
    );
}
