import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import { Separator } from '@/components/ui/separator';
import AppLayout from '@/layouts/app-layout';
import { Head, router, useForm } from '@inertiajs/react';
import { ArrowLeft, Key } from 'lucide-react';
import { FormEvent, useState } from 'react';

interface User {
    id: number;
    name: string;
    email: string;
    role: string;
    access_type?: string;
}

interface Props {
    user: User;
}

export default function EditUser({ user }: Props) {
    const { data, setData, processing, errors } = useForm({
        name: user.name,
        email: user.email,
        role: user.role,
        access_web: user.access_type === 'web' || user.access_type === 'both',
        access_dashboard: user.access_type === 'dashboard' || user.access_type === 'both',
    });

    const [showPasswordForm, setShowPasswordForm] = useState(false);
    const [passwordData, setPasswordData] = useState({
        password: '',
        password_confirmation: '',
    });

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();

        // Convert checkbox values to access_type enum
        let access_type = 'both';
        if (data.role === 'contributor') {
            if (data.access_web && !data.access_dashboard) {
                access_type = 'web';
            } else if (!data.access_web && data.access_dashboard) {
                access_type = 'dashboard';
            } else if (data.access_web && data.access_dashboard) {
                access_type = 'both';
            }
        }

        router.put(`/admin/users/${user.id}`, {
            name: data.name,
            email: data.email,
            role: data.role,
            access_type,
        });
    };

    const handlePasswordUpdate = (e: FormEvent) => {
        e.preventDefault();
        router.put(`/admin/users/${user.id}/password`, passwordData, {
            onSuccess: () => {
                setPasswordData({ password: '', password_confirmation: '' });
                setShowPasswordForm(false);
            },
        });
    };

    return (
        <AppLayout>
            <Head title={`Edit User - ${user.name}`} />

            <div className="container mx-auto px-4 py-8 max-w-2xl">
                {/* Header */}
                <div className="mb-8">
                    <Button
                        variant="ghost"
                        className="mb-4"
                        onClick={() => window.history.back()}
                    >
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back
                    </Button>
                    <h1 className="text-4xl font-bold mb-2">Edit User</h1>
                    <p className="text-muted-foreground">
                        Update user information and permissions
                    </p>
                </div>

                {/* User Info Form */}
                <form onSubmit={handleSubmit} className="space-y-6">
                    <Card>
                        <CardHeader>
                            <CardTitle>User Information</CardTitle>
                            <CardDescription>
                                Update the user's basic information
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            {/* Name */}
                            <div className="space-y-2">
                                <Label htmlFor="name">
                                    Name <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="name"
                                    value={data.name}
                                    onChange={(e) => setData('name', e.target.value)}
                                    required
                                />
                                {errors.name && (
                                    <p className="text-sm text-red-500">{errors.name}</p>
                                )}
                            </div>

                            {/* Email */}
                            <div className="space-y-2">
                                <Label htmlFor="email">
                                    Email <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="email"
                                    type="email"
                                    value={data.email}
                                    onChange={(e) => setData('email', e.target.value)}
                                    required
                                />
                                {errors.email && (
                                    <p className="text-sm text-red-500">{errors.email}</p>
                                )}
                            </div>

                            {/* Role */}
                            <div className="space-y-2">
                                <Label htmlFor="role">
                                    Role <span className="text-red-500">*</span>
                                </Label>
                                <Select
                                    value={data.role}
                                    onValueChange={(value) => setData('role', value)}
                                    required
                                >
                                    <SelectTrigger>
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="user">User</SelectItem>
                                        <SelectItem value="contributor">Contributor</SelectItem>
                                        <SelectItem value="admin">Admin</SelectItem>
                                    </SelectContent>
                                </Select>
                                {errors.role && (
                                    <p className="text-sm text-red-500">{errors.role}</p>
                                )}
                            </div>

                            {/* Access Type - Only for Contributors */}
                            {data.role === 'contributor' && (
                                <div className="space-y-2">
                                    <Label>
                                        Access Type <span className="text-red-500">*</span>
                                    </Label>
                                    <div className="space-y-3">
                                        <div className="flex items-center space-x-2">
                                            <Checkbox
                                                id="access_web"
                                                checked={data.access_web}
                                                onCheckedChange={(checked) => setData('access_web', checked as boolean)}
                                            />
                                            <Label htmlFor="access_web" className="font-normal cursor-pointer">
                                                Web - Can access web category sections
                                            </Label>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <Checkbox
                                                id="access_dashboard"
                                                checked={data.access_dashboard}
                                                onCheckedChange={(checked) => setData('access_dashboard', checked as boolean)}
                                            />
                                            <Label htmlFor="access_dashboard" className="font-normal cursor-pointer">
                                                Dashboard - Can access dashboard category sections
                                            </Label>
                                        </div>
                                    </div>
                                    {'access_type' in errors && errors.access_type && (
                                        // @ts-expect-error - access_type is added dynamically from backend validation
                                        <p className="text-sm text-red-500">{String(errors.access_type)}</p>
                                    )}
                                    {!data.access_web && !data.access_dashboard && (
                                        <p className="text-sm text-amber-600">
                                            Please select at least one access type
                                        </p>
                                    )}
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Submit */}
                    <div className="flex gap-2 justify-end">
                        <Button
                            type="button"
                            variant="outline"
                            onClick={() => window.history.back()}
                        >
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Updating...' : 'Update User'}
                        </Button>
                    </div>
                </form>

                <Separator className="my-8" />

                {/* Password Change Section */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <Key className="mr-2 h-5 w-5" />
                            Change Password
                        </CardTitle>
                        <CardDescription>
                            Update the user's password
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        {!showPasswordForm ? (
                            <Button onClick={() => setShowPasswordForm(true)}>
                                Change Password
                            </Button>
                        ) : (
                            <form onSubmit={handlePasswordUpdate} className="space-y-4">
                                {/* New Password */}
                                <div className="space-y-2">
                                    <Label htmlFor="password">
                                        New Password <span className="text-red-500">*</span>
                                    </Label>
                                    <Input
                                        id="password"
                                        type="password"
                                        value={passwordData.password}
                                        onChange={(e) =>
                                            setPasswordData({ ...passwordData, password: e.target.value })
                                        }
                                        placeholder="••••••••"
                                        required
                                    />
                                </div>

                                {/* Confirm Password */}
                                <div className="space-y-2">
                                    <Label htmlFor="password_confirmation">
                                        Confirm Password <span className="text-red-500">*</span>
                                    </Label>
                                    <Input
                                        id="password_confirmation"
                                        type="password"
                                        value={passwordData.password_confirmation}
                                        onChange={(e) =>
                                            setPasswordData({
                                                ...passwordData,
                                                password_confirmation: e.target.value,
                                            })
                                        }
                                        placeholder="••••••••"
                                        required
                                    />
                                </div>

                                {/* Buttons */}
                                <div className="flex gap-2">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() => {
                                            setShowPasswordForm(false);
                                            setPasswordData({ password: '', password_confirmation: '' });
                                        }}
                                    >
                                        Cancel
                                    </Button>
                                    <Button type="submit">Update Password</Button>
                                </div>
                            </form>
                        )}
                    </CardContent>
                </Card>
            </div>
        </AppLayout>
    );
}
