import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Input } from '@/components/ui/input';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import AppLayout from '@/layouts/app-layout';
import type { Category, PaginatedSections, SectionFilters } from '@/types/section';
import { Head, router } from '@inertiajs/react';
import { ChevronDown, Copy, Eye, Filter, Search } from 'lucide-react';
import { Fragment, useState } from 'react';
import { toast } from 'sonner';

interface Props {
    sections: PaginatedSections;
    categories: Category[];
    filters: SectionFilters;
}

export default function SectionsIndex({ sections, categories, filters }: Props) {
    const [search, setSearch] = useState(filters.search || '');
    const [selectedCategory, setSelectedCategory] = useState(filters.category || 'all');
    const [selectedSort, setSelectedSort] = useState(filters.sort || 'latest');

    const handleCopyCode = async (sectionId: number, versionId: number, framework: string) => {
        try {
            const response = await fetch(`/sections/${sectionId}/versions/${versionId}/code`);
            const data = await response.json();

            await navigator.clipboard.writeText(data.code);
            toast.success(`${framework.toUpperCase()} code copied to clipboard!`);

            // Reload sections data to show updated copies_count
            router.reload({ only: ['sections'] });
        } catch {
            toast.error('Failed to copy code');
        }
    };

    const handleSearch = () => {
        router.get(
            '/sections',
            {
                search: search || undefined,
                category: selectedCategory !== 'all' ? selectedCategory : undefined,
                sort: selectedSort,
            },
            { preserveState: true }
        );
    };

    const handleCategoryChange = (value: string) => {
        setSelectedCategory(value);
        router.get(
            '/sections',
            {
                search: search || undefined,
                category: value !== 'all' ? value : undefined,
                sort: selectedSort,
            },
            { preserveState: true }
        );
    };

    const handleSortChange = (value: string) => {
        setSelectedSort(value);
        router.get(
            '/sections',
            {
                search: search || undefined,
                category: selectedCategory !== 'all' ? selectedCategory : undefined,
                sort: value,
            },
            { preserveState: true }
        );
    };

    const handlePageChange = (page: number) => {
        router.get(
            '/sections',
            {
                page,
                search: search || undefined,
                category: selectedCategory !== 'all' ? selectedCategory : undefined,
                sort: selectedSort,
            },
            { preserveState: true }
        );
    };

    return (
        <AppLayout>
            <Head title="Component Library" />

            <div className="container mx-auto px-4 py-8">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-4xl font-bold mb-2">Component Library</h1>
                    <p className="text-muted-foreground">
                        Browse thousands of ready-to-use UI components for your projects
                    </p>
                </div>

                {/* Filters */}
                <div className="mb-8 flex flex-col gap-4 md:flex-row md:items-center">
                    <div className="flex-1 flex gap-2">
                        <Input
                            placeholder="Search components..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            className="max-w-md"
                        />
                        <Button onClick={handleSearch} size="icon">
                            <Search className="h-4 w-4" />
                        </Button>
                    </div>

                    <div className="flex gap-2">
                        <Select value={selectedCategory} onValueChange={handleCategoryChange}>
                            <SelectTrigger className="w-[180px]">
                                <Filter className="h-4 w-4 mr-2" />
                                <SelectValue placeholder="Category" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="all">All Categories</SelectItem>
                                {categories.map((category) => (
                                    <SelectItem key={category.id} value={category.id.toString()}>
                                        {category.name}
                                    </SelectItem>
                                ))}
                            </SelectContent>
                        </Select>

                        <Select value={selectedSort} onValueChange={handleSortChange}>
                            <SelectTrigger className="w-[180px]">
                                <SelectValue placeholder="Sort by" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="latest">Latest</SelectItem>
                                <SelectItem value="popular">Most Popular</SelectItem>
                                <SelectItem value="most_copied">Most Copied</SelectItem>
                                <SelectItem value="oldest">Oldest</SelectItem>
                            </SelectContent>
                        </Select>
                    </div>
                </div>

                {/* Results count */}
                <div className="mb-4 text-sm text-muted-foreground">
                    Showing {sections.from} - {sections.to} of {sections.total} components
                </div>

                {/* Sections Grid */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                    {sections.data.map((section) => (
                        <Card
                            key={section.id}
                            className="hover:shadow-lg transition-shadow cursor-pointer"
                            onClick={() => router.visit(`/sections/${section.slug}`)}
                        >
                            <CardHeader>
                                {section.thumbnail ? (
                                    <img
                                        src={`/storage/${section.thumbnail}`}
                                        alt={section.name}
                                        className="w-full h-48 object-cover rounded-md mb-4"
                                    />
                                ) : (
                                    <div className="w-full h-48 bg-muted rounded-md mb-4 flex items-center justify-center">
                                        <span className="text-muted-foreground">No preview</span>
                                    </div>
                                )}
                                <CardTitle>{section.name}</CardTitle>
                                <CardDescription className="line-clamp-2">
                                    {section.description || 'No description available'}
                                </CardDescription>
                            </CardHeader>

                            <CardContent>
                                <div className="flex flex-wrap gap-2 mb-4">
                                    <Badge variant="secondary">{section.category?.name}</Badge>
                                    {section.tags?.slice(0, 2).map((tag) => (
                                        <Badge key={tag} variant="outline">
                                            {tag}
                                        </Badge>
                                    ))}
                                </div>

                                <div className="flex gap-2 text-sm text-muted-foreground">
                                    {section.versions && section.versions.length > 0 && (
                                        <div className="flex gap-1">
                                            {Array.from(
                                                new Set(section.versions.map((v) => v.framework))
                                            ).map((framework) => (
                                                <Badge key={framework} variant="outline" className="text-xs">
                                                    {framework}
                                                </Badge>
                                            ))}
                                        </div>
                                    )}
                                </div>
                            </CardContent>

                            <CardFooter className="flex justify-between items-center">
                                <div className="flex items-center gap-4 text-sm text-muted-foreground">
                                    <div className="flex items-center gap-1">
                                        <Eye className="h-4 w-4" />
                                        <span>{section.views_count}</span>
                                    </div>
                                    <div className="flex items-center gap-1">
                                        <Copy className="h-4 w-4" />
                                        <span>{section.copies_count}</span>
                                    </div>
                                </div>

                                {section.versions && section.versions.length > 0 && (
                                    <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button
                                                size="sm"
                                                variant="outline"
                                                onClick={(e) => e.stopPropagation()}
                                            >
                                                <Copy className="h-4 w-4 mr-2" />
                                                Copy Code
                                                <ChevronDown className="h-4 w-4 ml-2" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent align="end">
                                            {section.versions.map((version) => (
                                                <DropdownMenuItem
                                                    key={version.id}
                                                    onClick={(e) => {
                                                        e.stopPropagation();
                                                        handleCopyCode(section.id, version.id, version.framework);
                                                    }}
                                                >
                                                    <Copy className="h-4 w-4 mr-2" />
                                                    Copy {version.framework.toUpperCase()} ({version.styling})
                                                </DropdownMenuItem>
                                            ))}
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                                )}
                            </CardFooter>
                        </Card>
                    ))}
                </div>

                {/* Pagination */}
                {sections.last_page > 1 && (
                    <div className="flex justify-center gap-2">
                        <Button
                            variant="outline"
                            onClick={() => handlePageChange(sections.current_page - 1)}
                            disabled={sections.current_page === 1}
                        >
                            Previous
                        </Button>

                        <div className="flex items-center gap-2">
                            {Array.from({ length: sections.last_page }, (_, i) => i + 1)
                                .filter(
                                    (page) =>
                                        page === 1 ||
                                        page === sections.last_page ||
                                        Math.abs(page - sections.current_page) <= 2
                                )
                                .map((page, index, array) => (
                                    <Fragment key={page}>
                                        {index > 0 && array[index - 1] !== page - 1 && (
                                            <span key={`ellipsis-${page}`}>...</span>
                                        )}
                                        <Button
                                            variant={page === sections.current_page ? 'default' : 'outline'}
                                            onClick={() => handlePageChange(page)}
                                        >
                                            {page}
                                        </Button>
                                    </Fragment>
                                ))}
                        </div>

                        <Button
                            variant="outline"
                            onClick={() => handlePageChange(sections.current_page + 1)}
                            disabled={sections.current_page === sections.last_page}
                        >
                            Next
                        </Button>
                    </div>
                )}
            </div>
        </AppLayout>
    );
}
