import { ConfirmationDialog } from '@/components/confirmation-dialog';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import AppLayout from '@/layouts/app-layout';
import type { PaginatedSections } from '@/types/section';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Copy, Edit, Eye, Package, Plus, RefreshCw, Trash2 } from 'lucide-react';
import { Fragment, useEffect, useState } from 'react';
import { toast } from 'sonner';

interface Props {
    sections: PaginatedSections;
}

export default function MySections({ sections }: Props) {
    const { props } = usePage<{ success?: string }>();
    const [deleteDialog, setDeleteDialog] = useState<{
        open: boolean;
        sectionId: number | null;
        sectionName: string;
    }>({
        open: false,
        sectionId: null,
        sectionName: '',
    });
    const [resubmitDialog, setResubmitDialog] = useState<{
        open: boolean;
        sectionId: number | null;
        sectionName: string;
    }>({
        open: false,
        sectionId: null,
        sectionName: '',
    });

    // Show flash messages
    useEffect(() => {
        if (props.success) {
            toast.success(props.success);
        }
    }, [props.success]);

    const handleDeleteClick = (sectionId: number, sectionName: string) => {
        setDeleteDialog({
            open: true,
            sectionId,
            sectionName,
        });
    };

    const handleDeleteConfirm = () => {
        if (!deleteDialog.sectionId) return;

        router.delete(`/sections/${deleteDialog.sectionId}`, {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Section deleted successfully!');
            },
            onError: () => {
                toast.error('Failed to delete section');
            },
        });
    };

    const handleResubmitClick = (sectionId: number, sectionName: string) => {
        setResubmitDialog({
            open: true,
            sectionId,
            sectionName,
        });
    };

    const handleResubmitConfirm = () => {
        if (!resubmitDialog.sectionId) return;

        router.post(`/sections/${resubmitDialog.sectionId}/resubmit`, {}, {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Section resubmitted for approval!');
            },
            onError: () => {
                toast.error('Failed to resubmit section');
            },
        });
    };

    const getStatusBadge = (status: string) => {
        switch (status) {
            case 'approved':
                return <Badge className="bg-green-500">Approved</Badge>;
            case 'pending':
                return <Badge className="bg-yellow-500">Pending</Badge>;
            case 'rejected':
                return <Badge className="bg-red-500">Rejected</Badge>;
            default:
                return <Badge>{status}</Badge>;
        }
    };



    return (
        <AppLayout>
            <Head title="My Sections" />

            <div className="container mx-auto px-4 py-8">
                {/* Header */}
                <div className="mb-8 flex items-center justify-between">
                    <div>
                        <h1 className="text-4xl font-bold mb-2">My Sections</h1>
                        <p className="text-muted-foreground">
                            Manage your submitted UI components
                        </p>
                    </div>
                    <Link href="/sections/create">
                        <Button>
                            <Plus className="mr-2 h-4 w-4" />
                            Add New Section
                        </Button>
                    </Link>
                </div>

                {/* Stats */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Total</CardDescription>
                            <CardTitle className="text-3xl">{sections.total}</CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Approved</CardDescription>
                            <CardTitle className="text-3xl text-green-500">
                                {sections.data.filter(s => s.status === 'approved').length}
                            </CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Pending</CardDescription>
                            <CardTitle className="text-3xl text-yellow-500">
                                {sections.data.filter(s => s.status === 'pending').length}
                            </CardTitle>
                        </CardHeader>
                    </Card>
                    <Card>
                        <CardHeader className="pb-2">
                            <CardDescription>Rejected</CardDescription>
                            <CardTitle className="text-3xl text-red-500">
                                {sections.data.filter(s => s.status === 'rejected').length}
                            </CardTitle>
                        </CardHeader>
                    </Card>
                </div>

                {/* Sections Grid */}
                {sections.data.length > 0 ? (
                    <>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                            {sections.data.map((section) => (
                                <Card key={section.id}>
                                    <CardHeader>
                                        {section.thumbnail ? (
                                            <img
                                                src={`/storage/${section.thumbnail}`}
                                                alt={section.name}
                                                className="w-full h-48 object-cover rounded-md mb-4"
                                            />
                                        ) : (
                                            <div className="w-full h-48 bg-muted rounded-md mb-4 flex items-center justify-center">
                                                <span className="text-muted-foreground">No preview</span>
                                            </div>
                                        )}
                                        <div className="flex items-start justify-between gap-2">
                                            <CardTitle className="flex-1">{section.name}</CardTitle>
                                            {getStatusBadge(section.status)}
                                        </div>
                                        <CardDescription className="line-clamp-2">
                                            {section.description || 'No description available'}
                                        </CardDescription>
                                    </CardHeader>

                                    <CardContent>
                                        <div className="flex flex-wrap gap-2 mb-4">
                                            <Badge variant="secondary">{section.category?.name}</Badge>
                                            {section.versions && section.versions.length > 0 && (
                                                <div className="flex gap-1">
                                                    {Array.from(
                                                        new Set(section.versions.map((v) => v.framework))
                                                    ).map((framework) => (
                                                        <Badge key={framework} variant="outline" className="text-xs">
                                                            {framework}
                                                        </Badge>
                                                    ))}
                                                </div>
                                            )}
                                        </div>

                                        {section.rejection_reason && (
                                            <div className="p-3 bg-red-50 dark:bg-red-950 border border-red-200 dark:border-red-800 rounded-md mb-4">
                                                <p className="text-sm font-medium text-red-900 dark:text-red-100 mb-1">
                                                    Rejection Reason:
                                                </p>
                                                <p className="text-sm text-red-700 dark:text-red-300">
                                                    {section.rejection_reason}
                                                </p>
                                            </div>
                                        )}
                                    </CardContent>

                                    <CardFooter className="flex justify-between items-center">
                                        <div className="flex items-center gap-4 text-sm text-muted-foreground">
                                            <div className="flex items-center gap-1">
                                                <Eye className="h-4 w-4" />
                                                <span>{section.views_count}</span>
                                            </div>
                                            <div className="flex items-center gap-1">
                                                <Copy className="h-4 w-4" />
                                                <span>{section.copies_count}</span>
                                            </div>
                                            {getStatusBadge(section.status)}
                                        </div>

                                        <div className="flex gap-2">
                                            <Link href={`/sections/${section.slug}`}>
                                                <Button size="sm" variant="outline">
                                                    <Eye className="h-4 w-4" />
                                                </Button>
                                            </Link>

                                            <Link href={`/sections/${section.id}/edit`}>
                                                <Button size="sm" variant="outline">
                                                    <Edit className="h-4 w-4" />
                                                </Button>
                                            </Link>

                                            {section.status === 'rejected' && (
                                                <Button
                                                    size="sm"
                                                    variant="default"
                                                    onClick={() => handleResubmitClick(section.id, section.name)}
                                                >
                                                    <RefreshCw className="h-4 w-4 mr-2" />
                                                    Resubmit
                                                </Button>
                                            )}

                                            <Button
                                                size="sm"
                                                variant="destructive"
                                                onClick={() => handleDeleteClick(section.id, section.name)}
                                            >
                                                <Trash2 className="h-4 w-4" />
                                            </Button>
                                        </div>
                                    </CardFooter>
                                </Card>
                            ))}
                        </div>

                        {/* Pagination */}
                        {sections.last_page > 1 && (
                            <div className="flex justify-center gap-2">
                                <Button
                                    variant="outline"
                                    onClick={() => router.get('/my-sections', { page: sections.current_page - 1 })}
                                    disabled={sections.current_page === 1}
                                >
                                    Previous
                                </Button>

                                <div className="flex items-center gap-2">
                                    {Array.from({ length: sections.last_page }, (_, i) => i + 1)
                                        .filter(
                                            (page) =>
                                                page === 1 ||
                                                page === sections.last_page ||
                                                Math.abs(page - sections.current_page) <= 2
                                        )
                                        .map((page, index, array) => (
                                            <Fragment key={page}>
                                                {index > 0 && array[index - 1] !== page - 1 && (
                                                    <span key={`ellipsis-${page}`}>...</span>
                                                )}
                                                <Button
                                                    variant={page === sections.current_page ? 'default' : 'outline'}
                                                    onClick={() => router.get('/my-sections', { page })}
                                                >
                                                    {page}
                                                </Button>
                                            </Fragment>
                                        ))}
                                </div>

                                <Button
                                    variant="outline"
                                    onClick={() => router.get('/my-sections', { page: sections.current_page + 1 })}
                                    disabled={sections.current_page === sections.last_page}
                                >
                                    Next
                                </Button>
                            </div>
                        )}
                    </>
                ) : (
                    <Card>
                        <CardContent className="flex flex-col items-center justify-center py-12">
                            <Package className="h-16 w-16 text-muted-foreground mb-4" />
                            <h3 className="text-xl font-semibold mb-2">No sections yet</h3>
                            <p className="text-muted-foreground mb-4">
                                You haven't created any sections. Start by adding your first component!
                            </p>
                            <Link href="/sections/create">
                                <Button>
                                    <Plus className="mr-2 h-4 w-4" />
                                    Create Your First Section
                                </Button>
                            </Link>
                        </CardContent>
                    </Card>
                )}
            </div>

            <ConfirmationDialog
                open={deleteDialog.open}
                onOpenChange={(open) => setDeleteDialog({ ...deleteDialog, open })}
                onConfirm={handleDeleteConfirm}
                title="Delete Section"
                description={`Are you sure you want to delete "${deleteDialog.sectionName}"? This action cannot be undone.`}
                confirmText="Delete"
                cancelText="Cancel"
                variant="destructive"
            />

            <ConfirmationDialog
                open={resubmitDialog.open}
                onOpenChange={(open) => setResubmitDialog({ ...resubmitDialog, open })}
                onConfirm={handleResubmitConfirm}
                title="Resubmit Section"
                description={`Resubmit "${resubmitDialog.sectionName}" for approval? This will notify the administrators to review your section again.`}
                confirmText="Resubmit"
                cancelText="Cancel"
                variant="default"
            />
        </AppLayout>
    );
}
