import { Head, router } from '@inertiajs/react';
import { ArrowLeft, Check, Copy, Download, Eye } from 'lucide-react';
import { useEffect, useState } from 'react';
import { toast } from 'sonner';

import { CodeBlock } from '@/components/code-block';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import AppLayout from '@/layouts/app-layout';
import type { Framework, Section, Styling } from '@/types/section';

interface Props {
    section: Section;
}

export default function SectionShow({ section }: Props) {
    const [selectedFramework, setSelectedFramework] = useState<Framework>(() => {
        // Initialize with the first available framework
        return section.versions?.[0]?.framework || 'html';
    });
    const [selectedStyling, setSelectedStyling] = useState<Styling>(() => {
        // Initialize with the first available styling
        return section.versions?.[0]?.styling || 'tailwind';
    });
    const [activeTab, setActiveTab] = useState<'preview' | 'code'>('preview');
    const [copied, setCopied] = useState(false);
    const [appCss, setAppCss] = useState<string>('');

    // Fetch the app CSS content
    useEffect(() => {
        const cssLink = document.querySelector('link[rel="stylesheet"][href*="/build/assets/app-"]') as HTMLLinkElement;
        console.log('CSS Link found:', cssLink?.href);

        if (cssLink) {
            fetch(cssLink.href)
                .then(res => {
                    console.log('CSS fetch response:', res.status);
                    return res.text();
                })
                .then(css => {
                    console.log('CSS loaded, length:', css.length);
                    setAppCss(css);
                })
                .catch(err => {
                    console.error('Failed to load CSS:', err);
                });
        } else {
            console.warn('No CSS link found, using Tailwind CDN fallback');
        }
    }, []);

    const currentVersion = section.versions?.find(
        (v) => v.framework === selectedFramework && v.styling === selectedStyling
    );

    const availableFrameworks = Array.from(
        new Set(section.versions?.map((v) => v.framework) || [])
    ) as Framework[];

    const availableStylings = Array.from(
        new Set(
            section.versions
                ?.filter((v) => v.framework === selectedFramework)
                .map((v) => v.styling) || []
        )
    ) as Styling[];

    const handleCopyCode = async () => {
        if (!currentVersion) return;

        try {
            await navigator.clipboard.writeText(currentVersion.code);
            setCopied(true);
            toast.success('Code copied to clipboard!');

            // Track copy and reload page data to get updated count
            await fetch(`/sections/${section.id}/versions/${currentVersion.id}/code`);
            router.reload({ only: ['section'] });

            setTimeout(() => setCopied(false), 2000);
        } catch {
            toast.error('Failed to copy code');
        }
    };

    const handleDownload = () => {
        if (!currentVersion) return;

        let fileExtension = 'html';
        if (selectedFramework === 'react') {
            fileExtension = 'tsx';
        } else if (selectedFramework === 'vue') {
            fileExtension = 'vue';
        } else if (selectedFramework === 'angular') {
            fileExtension = 'ts';
        }

        const blob = new Blob([currentVersion.code], { type: 'text/plain' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `${section.slug}.${fileExtension}`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);

        toast.success('Code downloaded!');
    };
    console.log(section.category)
    return (
        <AppLayout>
            <Head title={section.name} />

            <div className="container mx-auto px-4 py-8">
                {/* Back Button */}
                <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => window.history.back()}
                    className="mb-4"
                >
                    <ArrowLeft className="h-4 w-4 mr-2" />
                    Back
                </Button>

                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-start justify-between mb-4">
                        <div>
                            <h1 className="text-4xl font-bold mb-2">{section.name}</h1>
                            <p className="text-muted-foreground">{section.description}</p>
                        </div>
                        <div className="flex gap-2 text-sm text-muted-foreground">
                            <div className="flex items-center gap-1">
                                <Eye className="h-4 w-4" />
                                <span>{section.views_count} views</span>
                            </div>
                            <div className="flex items-center gap-1">
                                <Copy className="h-4 w-4" />
                                <span>{section.copies_count} copies</span>
                            </div>
                        </div>
                    </div>

                    <div className="flex flex-wrap gap-2">
                        <Badge variant="secondary">{section.category?.name}</Badge>
                        {section.tags?.map((tag) => (
                            <Badge key={tag} variant="outline">
                                {tag}
                            </Badge>
                        ))}
                    </div>

                    {section.user && (
                        <div className="mt-4 text-sm text-muted-foreground">
                            Created by <span className="font-medium">{section.user.name}</span>
                        </div>
                    )}
                </div>

                {/* Framework and Styling Selectors */}
                <div className="mb-6 flex flex-wrap justify-between gap-4">
                    <div>
                        <label className="text-sm font-medium mb-2 block">Framework</label>
                        <div className="flex gap-2">
                            {availableFrameworks.map((framework) => (
                                <Button
                                    key={framework}
                                    size="sm"
                                    variant={selectedFramework === framework ? 'default' : 'outline'}
                                    onClick={() => {
                                        setSelectedFramework(framework);
                                        // Reset styling if not available for new framework
                                        const stylings = Array.from(
                                            new Set(
                                                section.versions
                                                    ?.filter((v) => v.framework === framework)
                                                    .map((v) => v.styling) || []
                                            )
                                        );
                                        if (!stylings.includes(selectedStyling)) {
                                            setSelectedStyling(stylings[0] || 'tailwind');
                                        }
                                    }}
                                >
                                    {framework.charAt(0).toUpperCase() + framework.slice(1)}
                                </Button>
                            ))}
                        </div>
                    </div>

                    <div>
                        <label className="text-sm font-medium mb-2 block">Styling</label>
                        <div className="flex gap-2">
                            {availableStylings.map((styling) => (
                                <Button
                                    key={styling}
                                    variant={selectedStyling === styling ? 'default' : 'outline'}
                                    size="sm"
                                    onClick={() => setSelectedStyling(styling)}
                                >
                                    {styling.charAt(0).toUpperCase() + styling.slice(1)}
                                </Button>
                            ))}
                        </div>
                    </div>
                </div>

                {/* Preview and Code Tabs */}
                <Card>
                    <CardHeader>
                        <div className="flex items-center justify-between">
                            <div>
                                <CardTitle>
                                    {selectedFramework.charAt(0).toUpperCase() + selectedFramework.slice(1)} +{' '}
                                    {selectedStyling.charAt(0).toUpperCase() + selectedStyling.slice(1)}
                                </CardTitle>
                                <CardDescription>
                                    {currentVersion ? 'Version ' + currentVersion.version : 'Not available'}
                                </CardDescription>
                            </div>
                            <div className="flex gap-2">
                                <Button variant="outline" size="sm" onClick={handleDownload} disabled={!currentVersion}>
                                    <Download className="h-4 w-4 mr-2" />
                                    Download
                                </Button>
                                <Button size="sm" onClick={handleCopyCode} disabled={!currentVersion}>
                                    {copied ? (
                                        <>
                                            <Check className="h-4 w-4 mr-2" />
                                            Copied!
                                        </>
                                    ) : (
                                        <>
                                            <Copy className="h-4 w-4 mr-2" />
                                            Copy Code
                                        </>
                                    )}
                                </Button>
                            </div>
                        </div>
                    </CardHeader>

                    <CardContent>
                        {currentVersion ? (
                            <Tabs value={activeTab} onValueChange={(v) => setActiveTab(v as 'preview' | 'code')}>
                                <TabsList className="mb-4">
                                    <TabsTrigger value="preview">Preview</TabsTrigger>
                                    <TabsTrigger value="code">Code</TabsTrigger>
                                </TabsList>

                                <TabsContent value="preview">
                                    <div className="border rounded-lg bg-background min-h-[400px]">
                                        {selectedFramework === 'html' ? (
                                            <iframe
                                                srcDoc={`
                                                    <!DOCTYPE html>
                                                    <html>
                                                    <head>
                                                        <meta charset="UTF-8">
                                                        <meta name="viewport" content="width=device-width, initial-scale=1.0">
                                                        ${appCss ? `<style>${appCss}</style>` : '<script src="https://cdn.tailwindcss.com"></script>'}
                                                        <style>
                                                            body { 
                                                                margin: 0; 
                                                                padding: 0;
                                                                min-height: 100vh;
                                                            }
                                                        </style>
                                                    </head>
                                                    <body>
                                                        ${currentVersion.code}
                                                    </body>
                                                    </html>
                                                `}
                                                className="w-full h-[600px] border-0 rounded-lg"
                                                title="Component Preview"
                                                sandbox="allow-scripts"
                                            />
                                        ) : (
                                            <div className="flex flex-col items-center justify-center h-[400px] text-center p-8">
                                                <div className="text-muted-foreground mb-4">
                                                    <svg
                                                        className="w-16 h-16 mx-auto mb-4"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth={2}
                                                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                                                        />
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth={2}
                                                            d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                                                        />
                                                    </svg>
                                                    <h3 className="text-lg font-semibold mb-2">
                                                        Preview not available for {selectedFramework.toUpperCase()}
                                                    </h3>
                                                    <p className="text-sm max-w-md">
                                                        {selectedFramework === 'react' &&
                                                            'React components require a build environment. Copy the code and use it in your React project.'}
                                                        {selectedFramework === 'vue' &&
                                                            'Vue components require a build environment. Copy the code and use it in your Vue project.'}
                                                        {selectedFramework === 'angular' &&
                                                            'Angular components require a build environment. Copy the code and use it in your Angular project.'}
                                                    </p>
                                                </div>
                                                <Button
                                                    variant="outline"
                                                    onClick={() => setActiveTab('code')}
                                                    className="mt-4"
                                                >
                                                    View Code Instead
                                                </Button>
                                            </div>
                                        )}
                                    </div>
                                </TabsContent>

                                <TabsContent value="code">
                                    <CodeBlock
                                        code={currentVersion.code}
                                        language={
                                            selectedFramework === 'html' ? 'html' :
                                                selectedFramework === 'react' ? 'tsx' :
                                                    selectedFramework === 'vue' ? 'vue' :
                                                        selectedFramework === 'angular' ? 'typescript' :
                                                            'html'
                                        }
                                        theme="github-dark"
                                        showLineNumbers={true}
                                    />

                                    {currentVersion.dependencies && currentVersion.dependencies.length > 0 && (
                                        <div className="mt-4">
                                            <h3 className="font-medium mb-2">Dependencies</h3>
                                            <div className="flex flex-wrap gap-2">
                                                {currentVersion.dependencies.map((dep) => (
                                                    <Badge key={dep} variant="secondary">
                                                        {dep}
                                                    </Badge>
                                                ))}
                                            </div>
                                        </div>
                                    )}

                                    {currentVersion.notes && (
                                        <div className="mt-4">
                                            <h3 className="font-medium mb-2">Notes</h3>
                                            <p className="text-sm text-muted-foreground">{currentVersion.notes}</p>
                                        </div>
                                    )}
                                </TabsContent>
                            </Tabs>
                        ) : (
                            <div className="text-center py-12 text-muted-foreground">
                                This combination of framework and styling is not available for this component.
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </AppLayout>
    );
}
